/* 
 * intersect.cpp: Sample programs of ray-object intersections
 */
#include <math.h>
#include "Ray.h"
#include "Plane.h"
#include "Box.h"
#include "Triangle.h"
#include "Sphere.h"

using namespace std;

bool intersect_ray_plane(const Ray &ray, const Plane &plane, double &t)
{
  t = 0;
  if ( ray.O == plane.p0 ) 
    return true;
  
  Vector3 origin_to_p0 = plane.p0 - ray.O;  //vector from origin to p0 
  double op_dot_n = origin_to_p0 * plane.n;      //dot product
  double d_dot_n = ray.D * plane.n;      //dot product

  if (  fabs (d_dot_n)  < 0.0000001 )	//ray parallel or pointing downward
    return false;

  t =  op_dot_n / d_dot_n;
  if ( t < 0.0 )
    return false;

  return true;
}

bool intersect_ray_box(Ray &ray, Box &box, double &t1, double &t2)
{
  double t_near = -1.0e+10, t_far = 1.0e+10;
  double D[3];  //ray direction
  double O[3];  //ray origin
  double pl[3]; //minimum extent
  double ph[3]; //maximum extent
  ray.D.getXYZ ( D );  //put ray direction (x,y,z) in D[]
  ray.O.getXYZ ( O ); //put ray origin (x,y,z) in O[]
  box.pl.getXYZ ( pl );  //put (x,y,z) of minimum point in pl[]
  box.ph.getXYZ ( ph );  //put (x,y,z) of maximum point in ph[]
  
  for ( int i = 0; i < 3; i++ ) {
    if ( D[i] == 0 ) {   //ray parallel to an axis plane
      if ( O[i] < pl[i] || O[i] > ph[i] )
        return false;    //origin not between points
    } else {
       t1 = ( pl[i] - O[i] ) / D[i];
       t2 = ( ph[i] - O[i] ) / D[i];
       if ( t1 > t2 ) {  //swap t1, t2
         double temp = t1;
         t1 = t2;
         t2 = temp;
       }
       if ( t1 > t_near ) t_near = t1;   //find max (t_near)
       if ( t2 < t_far  ) t_far  = t2;   //find min (t_far)
       if (t_near > t_far) return false;
       if (t_far < 0 ) return false;
    } 
  } //for
  t1 = t_near;
  t2 = t_far;
  
  return true;   //ray hits box
}

//assume that x, y, z are positive
char max_axis ( float x, float y, float z )
{
  char a;

  if ( x  > y )
    if ( x > z )
	a = 'x';
    else
	a = 'z';
  else
    if ( y > z )
	a = 'y';
    else
	a = 'z';

  return a;
}

bool intersect_ray_triangle ( Ray &ray, const Triangle &triangle, double &t )
{
  Vector3 v01 = triangle.p1 - triangle.p0;
  Vector3 v02 = triangle.p2 - triangle.p0;
  Vector3 normal = v01 ^ v02;		//normal to triangle
  Plane plane ( triangle.p0, normal );	//plane containing triangle
  bool hit_plane = intersect_ray_plane ( ray, plane, t );
  if ( !hit_plane ) return false;

  Point3 q = ray.getPoint( t );
  //cout << "q = " << q << endl;

  //check if q is inside triangle
  double bx, by, cx, cy, hx, hy;
  double beta, gamma;
  Vector3 b, c, h;

  c = v01;
  b = v02;
  h = q - triangle.p0;

  //find the dominant axis
  char axis = max_axis ( fabs ( normal.x), fabs ( normal.y), fabs(normal.z) );
  
  switch ( axis ) {
    case 'x':		//project on X-plane (yz)
      bx = b.y;	by = b.z;
      cx = c.y; cy = c.z;
      hx = h.y; hy = h.z;
      break;
    case 'y':		//project on Y-plane (zx)
      bx = b.z;	by = b.x;
      cx = c.z; cy = c.x;
      hx = h.z; hy = h.x;
      break;
    case 'z':		//project on Z-plane (xy)
      bx = b.x;	by = b.y;
      cx = c.x; cy = c.y;
      hx = h.x; hy = h.y;
      break;
  }

  double denominator =  bx * cy - by * cx;
  beta = ( bx * hy - by * hx ) / denominator;
  gamma = ( hx * cy - hy * cx ) / denominator;

  if ( beta < 0 ) return false;
  if ( gamma < 0 ) return false;
  if ( 1 - (beta + gamma) < 0 ) return false;
 
  return true;
}

bool intersect_ray_sphere(const Ray& ray,  const Sphere &sphere, double &t)
{
    double r = sphere.R;

    //Compute a, b and c coefficients
    double a =  ray.D * ray.D;    //dot product
    Vector3 oc = ray.O - sphere.C;
   // cout << oc << endl;
    oc.print();
    //double b = 2 * dot(oc, ray.direction);
    double b = 2 * oc * ray.D;
   // double c = dot(oc, oc) - (r * r);
    double c = oc * oc - r * r;

    //Find discriminant
    double disc = b * b - 4 * a * c;
    
    // if discriminant is negative there are no real roots, so return 
    // false as ray misses sphere
    if (disc < 0)
        return false;

    // compute the roots 
    double distSqrt = sqrtf(disc);
    double t0 = (-b - distSqrt ) / ( 2 * a );
    double t1 = (-b + distSqrt ) / ( 2 * a );
    
    // make sure t0 is smaller than t1
    if (t0 > t1)
    {
        // if t0 is bigger than t1, swap them around
        double temp = t0;
        t0 = t1;
        t1 = temp;
    }

    // if t1 < 0, object is in the ray's negative direction
    // as a  consequence the ray misses the sphere
    if (t1 < 0)
        return false;

    // if t0 is less than zero, the intersection point is at t1
    if (t0 < 0)  {
        t = t1;
        return true;
    } else {  // else the intersection point is at t0
        t = t0;
        return true;
    }
}
