/*
 * spheremap.cpp:  Draw a sphere with texture images. The cube can be rotated
 * by pressing keys 'x', 'X', 'y', 'Y', 'z', 'Z'.
 * Images are downloaded from Internet.  
 *
 */

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "imageio.h"

int texImageWidth;
int texImageHeight;
int window;
static GLuint handles[6];		//texture names
int anglex= 0, angley = 0, anglez = 0;	//rotation angles


char maps[][20] = {"../data/earth.png"};


class Point3 {
public:
  double x;
  double y;
  double z;
  Point3()
  {
    x = y = z = 0.0;
  }

  Point3 ( double x0, double y0, double z0 )
  {
    x = x0;	y = y0;	   z = z0;
  }
}; 

//load texture image
GLubyte *makeTexImage( char *loadfile )
{
   int i, j, c, width, height;
   GLubyte *texImage;
  
   /*
     Only works for .png or .tif images.  NULL is returned if errors occurred.
     loadImageRGBA() is from imageio library discussed in Chapter 6.
   */ 
   texImage = loadImageRGBA( (char *) loadfile, &width, &height);	
   texImageWidth = width;
   texImageHeight = height;

   return texImage;
}

void init(void)
{    
   glClearColor (1, 1, 1, 0.0);
   glShadeModel(GL_FLAT);

   glEnable(GL_DEPTH_TEST);

   glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
   //handles is global
   glGenTextures(1, handles);

   for ( int i = 0; i < 1; ++i ) {	
    GLubyte *texImage = makeTexImage( maps[i] );
    if ( !texImage ) {
      printf("\nError reading %s \n", maps[i] );
      continue;
    }
    glBindTexture(GL_TEXTURE_2D, handles[i]);	//now we work on handles
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, texImageWidth, 
                texImageHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE, texImage);

    delete texImage;    //free memory holding texture image
  }
}

const double PI = 3.141592654;
const double TWOPI = 6.283185308;
/*
 *    Create a sphere centered at c, with radius r, and precision n
 *    Draw a point for zero radius spheres
 */
void createSphere(Point3 c,double r,int n)
{
   int i,j;
   double phi1, phi2, theta, u, v;
   Point3 p, q;
   
   if ( r < 0 || n < 0 )
      return;
   if (n < 4 || r <= 0) {
      glBegin(GL_POINTS);
      glVertex3f(c.x,c.y,c.z);
      glEnd();
      return;
   }

   for ( j=0; j < n; j++ ) {
      phi1 = j * TWOPI / n;
      phi2 = (j + 1) * TWOPI / n;	//next phi

      glBegin(GL_QUAD_STRIP);
      for ( i=0; i <= n; i++ ) {
         theta = i * PI / n;

        q.x = sin ( theta ) * cos ( phi2 );
	q.y = sin ( theta ) * sin ( phi2 );
        q.z = cos ( theta );
        p.x = c.x + r * q.x;
        p.y = c.y + r * q.y;
        p.z = c.z + r * q.z;

        glNormal3f ( q.x, q.y, q.z );
        u = (double)(j+1) / n;		// column
 	v = 1 - (double) i / n;		// row
        glTexCoord2f( u, v );
        glVertex3f( p.x, p.y, p.z );

        q.x = sin ( theta ) * cos ( phi1 );
	q.y = sin ( theta ) * sin ( phi1 );
        q.z = cos ( theta );
        p.x = c.x + r * q.x;
        p.y = c.y + r * q.y;
        p.z = c.z + r * q.z;

         glNormal3f ( q.x, q.y, q.z );
         u = (double) j / n;		// column
	 v = 1 - (double) i / n;	// row
         glTexCoord2f( j / (double) n, 1 - i / (double) n );
         glVertex3f ( p.x, p.y, p.z );
      }
      glEnd();
   }
}

void display(void)
{
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   glEnable(GL_TEXTURE_2D);
   glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
   glEnable( GL_CULL_FACE );
   glCullFace ( GL_BACK );
  
   glPushMatrix(); 
   glRotatef( anglex, 1.0, 0.0, 0.0);			//rotate the object along x-axis
   glRotatef( angley, 0.0, 1.0, 0.0);			//rotate along y-axis	
   glRotatef( anglez, 0.0, 0.0, 1.0);			//rotate along z-axis

   glPointSize ( 6 );
   Point3 c;

   createSphere( c, 3.0 , 40 );

   glPopMatrix();
   glFlush();
   glDisable(GL_TEXTURE_2D);
}

void keyboard(unsigned char key, int x, int y)
{
  switch(key) {
    case 'x':
      anglex = ( anglex + 3 ) % 360;
      break;
    case 'X':
      anglex = ( anglex - 3 ) % 360;
      break;
    case 'y':
      angley = ( angley + 3 ) % 360;
      break;
    case 'Y':
      angley = ( angley - 3 ) % 360;
      break;
    case 'z':
      anglez = ( anglez + 3 ) % 360;
      break;
    case 'Z':
      anglez = ( anglez - 3 ) % 360;
      break;
    case 'r':
      anglex = angley = anglez = 0;
      break;
    case 27: /* escape */
        glutDestroyWindow(window);
        exit(0);
  }
  glutPostRedisplay();
}

void reshape(int w, int h)
{
   glViewport(0, 0, (GLsizei) w, (GLsizei) h);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   //gluPerspective(60.0, (GLfloat) w/(GLfloat) h, 1.0, 30.0);
    glOrtho(-4.0, 4.0, -4.0 * (GLfloat) h / (GLfloat) w,
            4.0 * (GLfloat) h / (GLfloat) w, -10.0, 10.0);

   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   //gluLookAt ( 0, 0, 5, 0, 0, 0, 0, 1, 0 );
   gluLookAt ( 5, 0, 0, 0, 0, 0, 0, 0, 1 );
}


int main(int argc, char** argv)
{
   glutInit(&argc, argv);
   glutInitDisplayMode(GLUT_SINGLE | GLUT_RGB | GLUT_DEPTH);
   glutInitWindowSize(500, 500);
   glutInitWindowPosition(100, 100);
   window = glutCreateWindow( "The Earth" );
   init();
   glutDisplayFunc(display);
   glutReshapeFunc(reshape);
   glutKeyboardFunc(keyboard);
   glutMainLoop();
   return 0; 
}
