// Sphere.cpp : class for rendering a wireframe sphere

#include <math.h>
#include <GL/glew.h>
#include <GL/glu.h>
#include <GL/glut.h>
#include "Sphere.h"
#include <stdio.h>
#include <vector>
#include <string>

using namespace std;

// Set color of displaying object
// with red, green, blue and alpha (opacity) values
const float Sphere::color[] = { 0.9f, 0.1f, 0.9f, 1.0f }; 
    
// Create a Sphere object
Sphere::Sphere ( int  &success )
{
 string *vs, *fs;
  char *vsSource, *fsSource;

  // Read shader source code.
  readShaderFile( (char *) "sphere.vert",  &vsSource);
  readShaderFile( (char *)"sphere.frag",  &fsSource);
  vs = new string ( vsSource );
  fs = new string ( fsSource );
  success = createShader ( vs, fs );
  if ( !success ) {
    printf("infoLog: %s\n", infoLog );
    return;
  }
  delete vs;       delete fs;
  delete vsSource; delete fsSource;

  createSphere (1.0f, N_SLICES, N_STACKS );
  int nVertices = vertices.size();
  int k = 0;
  int m = N_STACKS;
  int n = N_SLICES;
  // 2n(m-1) slices + 2(m-2)n stacks
  drawOrderwSize = 4 * m * n - 6 * n;
  drawOrderw = new short[drawOrderwSize];
  //The slices
  for ( int j = 0; j < n; j++ ) {
    for ( int i = 0; i < m-1; i++ ) {
      drawOrderw[k++] = (short) (j * m + i);
      drawOrderw[k++] = (short)( j* m + i + 1 );
    }
  }
  //The stacks (no need to draw segments at poles)
  for ( int i = 1; i < m - 1; i++) {
    for ( int j = 0; j < n; j++){
      drawOrderw[k++] = (short) (j * m + i);
      if ( j == n - 1)  //wrap around: j + 1 --> 0
        drawOrderw[k++] = (short) ( i);
      else
        drawOrderw[k++] = (short) ((j+1)*m + i);
    }
  }

  sphereCoords = new float[3*nVertices];
  k = 0;
  for ( int i = 0; i < nVertices; i++ ) {
    XYZ v = vertices[i];
    sphereCoords[k++] = v.x;
    sphereCoords[k++] = v.y;
    sphereCoords[k++] = v.z;
  }
}

Sphere::~Sphere()
{
  while (vertices.size() > 0 )
    vertices.pop_back();
}
  
void Sphere::createSphere ( float r, int nSlices, int nStacks )
{
     double phi,  theta;
     XYZ *p = new XYZ();
     const double PI = 3.1415926;
     const double TWOPI = 2 * PI;

     for ( int j = 0; j < nSlices; j++ ) {
       phi = j * TWOPI / nSlices;
       for ( int i = 0; i < nStacks; i++ ) {
         theta = i * PI / (nStacks-1);  //0 to pi
             p->x = r * (float) ( sin ( theta ) *  cos ( phi ));
         p->y = r * (float) ( sin ( theta ) *  sin ( phi ));
         p->z = r * (float)  cos ( theta );
         vertices.push_back ( *p  );
       }
     }
}

void Sphere::draw( float mvpMatrix[4][4] ) 
{
  // get handle to vertex shader's attribute variable vPosition
  int positionHandle= glGetAttribLocation(program,"vPosition");

  // Enable a handle to the sphere vertices
  glEnableVertexAttribArray( positionHandle );

  // Prepare the sphere coordinate data
  glVertexAttribPointer(positionHandle, COORDS_PER_VERTEX,
                           GL_FLOAT, false, 0, sphereCoords);

  int mvpMatrixHandle = glGetUniformLocation(program, "mvpMatrix");
  if (mvpMatrixHandle == -1)
     printf("No such uniform named %s\n", "mvpMatrix" );
  // pass model-view projection matrix to vertex shader
  glUniformMatrix4fv(mvpMatrixHandle, 1, GL_FALSE, &mvpMatrix[0][0] );

  // get handle to fragment shader's uniform variable vColor
  int colorHandle =  glGetUniformLocation(program, "vColor");
  if ( colorHandle == -1 )
    printf("No such uniform named vColor\n");

  // Set color for drawing the sphere
  glUniform4fv ( colorHandle, 1, color );
  glDrawElements( GL_LINES, drawOrderwSize,
                              GL_UNSIGNED_SHORT, drawOrderw);  
  // set blue color  
  glUniform4f ( colorHandle, 0.0, 0.0, 1.0, 1.0 );
  // Draw the north pole
  glDrawElements( GL_POINTS, 1,
                              GL_UNSIGNED_SHORT, drawOrderw);      
  // Disable vertex array
  glDisableVertexAttribArray(positionHandle);
}

