/*
 * seashell.cpp
 * Create seashell using extrusion and time-varying matrix.
 * C functions of vector operators are defined in Vector3.cpp of Chapter 12
 * Fore June
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <GL/glut.h>
#include <iostream>
#include <vector>
#include <algorithm>
#include <math.h>
#include "Point3.h"   //in ../12
#include "Vector3.h"  //in ../12
#include "Matrix4.h"
#include "Matrix14.h"
#include "Matrix41.h"
#include "Matrix44.h"

using namespace std;

const double PI = 3.14159265389;

//The curve
void get_C ( float C[4], float t, float b )
{
  C[0] = cos ( t );
  C[1] = sin ( t );
  C[2] = b * t;
  C[3] = 1;
}

//The tangent
void get_T ( double T[4], double C[4], double  u, double r, double p, double q )
{
  double  t1, t2;
  t1 = q * u;
  t2 = p * u;
  T[0] = -p * C[1] - r * q * sin ( t1 ) * cos ( t2 );
  T[1] = p * C[0] - r * q * sin ( t1 ) * sin ( t2 );
  T[2] = r * q * cos ( t1 );
  T[3] = 0;
}

//The principal normal
void get_N ( double N[4],  double C[4], double T[4], double u,  double r, double p, double q )
{
  float t1, t2;
  t1 = q * u;
  t2 = p * u;

  N[0] = -p * T[1] + r * q * ( p * sin(t1) * sin(t2) - q * cos(t1) * cos(t2) );
  N[1] = p * T[0] - r * q * ( p * sin(t1) * cos(t2) + q* cos(t1)*sin(t2) );
  N[2] = -q * q * r * sin ( t1 );
  N[3] = 0;
}

//The time-dependent matrix
void setMt ( Matrix44 &M, float u )
{
  M.setCol( 0, 1, 0, 0, 0 );
  M.setCol( 1,  0, u / 10, 0, 0 );
  M.setCol( 2,  0, 0,  u/10, 0 );
  M.setCol( 3,  0, 0,  0, 1 );
}

//The basic transformation matrix
void setM( Matrix44 &M, float u, float b )
{
  float c = 1.0 / sqrt ( 1 + b*b );
  M.setCol( 0, -sin(u)*c, cos(u) *c, b * c, 0); //Tangent   T(t)
  M.setCol( 1,  -cos(u), -sin(u), 0, 0 );       //Normal    N(t)
  M.setCol( 2, sin(u)*b*c, -cos(u)*b*c, c, 0 ); //Binormal  B(t)
  M.setCol( 3, cos(u), sin(u), b*u, 1 );        //The curve C(t)
}

class Cdouble3 {    //Note: array is not copyable; e.g. int a[8],b[8]; "a = b;" won't work 
  public:
     double p3[3];
};


void seaShell(void)
{
  const float b = 0.15;   //constant of Helix curve
  double H = 6.0;
  Matrix44 M44;           //Transformation matrix
  Matrix44 Mt;            //Matrix depends on parameter u
  const int N = 16;       //number of vertices in base
  float C[4];
  vector<Cdouble3>vp0(N), vp1(N);
  Matrix41 p_1;           //transformed po
  Matrix41 points[N];     //define N points
  double a = 0.15;
  double x, y, z;
  double dtheta = 2 * 3.1415926/ N;
  double theta = 0;
  for ( int i = 0; i < N; i++ ) {
    z = a *   cos ( theta );
    y = a *  sin ( theta );
    x = 0;
    points[i] = Matrix41( x, y, z, 1 );
    theta += dtheta;
  }
  glLineWidth ( 3 );
  glBegin(GL_LINE_STRIP );
  for ( float u = 0; u <=  18; u += 0.2 ) {
     get_C ( C, u, b );
     glVertex4fv( C );
  }
  glEnd();

  float p3[3];          //3-D point, (x, y, z)
  setM ( M44, 0, b );
  setMt ( Mt, 0 );
  for ( int i = 0; i < N; ++i ) {
    p_1 = M44 * Mt * points[i]; //transform the point
    p_1.get3( vp0[i].p3 );      //put (x, y, z) in vp0[i].p3[]
  }
  double normal[3], va[3], vb[3], vc[3];
  glBegin( GL_QUADS );          //a side has four points
  for ( float t = 0.05; t <=  32; t += 0.05 ) {
    setMt ( Mt, t );
    setM ( M44, t, b );
    for ( int i = 0; i < N; ++i ) {
      p_1 = M44 * Mt *  points[i];    //transform the point
      p_1.get3( vp1[i].p3 );          //put (x, y, z) in vp1[i].p3[]
    }
    for ( int i = 0; i < N; ++i ) {   //draw the N sides of tube between 'base' and 'cap'
      int j = (i+1) % N;
      for ( int ii = 0; ii < 3; ii++ ) {
         va[ii] = vp0[i].p3[ii];
         vb[ii] = vp0[j].p3[ii];
         vc[ii] = vp1[i].p3[ii];
      }
      plane_normal ( normal, va, vb, vc );
      glNormal3dv ( normal );
      glVertex3dv( vp0[i].p3 );
      glVertex3dv( vp0[j].p3 );
      glVertex3dv( vp1[j].p3 );
      glVertex3dv( vp1[i].p3 );
    }
    copy ( vp1.begin(), vp1.end(), vp0.begin() );       //copy vp1 to vp0
  }  //for u 
  glEnd();
}
