/*
 * twocenter.cpp
 * Demonstration of using two-center method to create anaglyph.
 * Fore June
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <GL/glut.h>
#include <iostream>

using namespace std;

void drawScene(void);

class Point3{
public:
  double x, y, z;

  Point3()
  { x = y = z = 0; }

  Point3 ( double x0, double y0, double z0 )
  {
    x = x0;  y = y0;  z = z0;
  }  
  Point3 ( const Point3 &p )
  {
    x = p.x;    y = p.y;   z = p.z;
  }
};

class Vector3{
public:
  double x, y, z;

  Vector3()
  { x = y = z = 0; }

  Vector3 ( double x0, double y0, double z0 )
  {
    x = x0;  y = y0;  z = z0;
  }

  Vector3 ( const Vector3 &v )
  {
    x = v.x;    y = v.y;    z = v.z;
  }
};

class Camera {
public:
   Point3 p;            // View position          
   Point3 focus;        // Point at which camera focuses
   Vector3   v;            // View direction vector   
   Vector3  up;            // View up direction      
   double f;            // Focal Length along v
   double fov;          // Camera aperture (field of view)         
   double es;           // Eye separation   
   int    w;            // Viewplane width
   int    h;            // Viewplane height       

   //constructors
   Camera ()
   {
      // default parameters
      f = 10.0;
      es = f / 30;
      fov = 60;
      w = 400;
      h = 300;
      p = Point3 ( 0, 0, -10 );  //viewpoint
      focus = Point3( 0, 0, 0); //focus
      v = Vector3 ( 0, 0, 1 );     //view direction
      up = Vector3 ( 0, 1, 0 );    //up vector
   }

   void setCamera ( double f0, double es0, double fov0, int w0, int h0 )
   {
     f  = f0;
     es = es0;
     fov = fov0;
     w  = w0;
     h  = h0;
   }

   void lookAt ()
   {
     gluLookAt( p.x, p.y, p.z, focus.x, focus.y, focus.z, up.x, up.y, up.z );
   }
   void lookAt ( const Point3 &eyePos, const Point3 &focus, const Vector3 &up0 )
   {
     p = eyePos;
     up = up0;
     gluLookAt( p.x, p.y, p.z, focus.x, focus.y, focus.z, up.x, up.y, up.z );
   }
};

Camera camera;

/*
   Initialization.
   This is where global OpenGL/GLUT settings are made. 
*/
void init(void)
{
   glEnable(GL_DEPTH_TEST);
   glClearColor(0.0,0.0,0.0,0.0);
   glClearAccum(0.0,0.0,0.0,0.0);   // The default 
}

/*
   This is the basic display callback routine.
   It creates the geometry, viewing position, and focus.
   It renders left eye first, then right.
*/
void display(void)
{
   /*
     L = left, R = right, T = top, B = bottom, N = near, F = far, f = focal length
     e = eye separation, a = half-width of projection plane, ratio = aspect ratio
     theta2 = (field of view)/2
   */
   double theta2, near,far;
   double L, R, T, B, N, F, f, e, a, b, c, ratio;

   // Clip to avoid extreme stereo 
   near = camera.f / 5;
   far = 1000.0;
   f = camera.f;

   // Set the buffer for writing and reading 
   glDrawBuffer(GL_BACK);
   glReadBuffer(GL_BACK);

   // Clear things 
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   glClear(GL_ACCUM_BUFFER_BIT); 

   // Left eye filter
   glColorMask(GL_TRUE,GL_FALSE,GL_FALSE,GL_TRUE);

   // Create the projection 
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   theta2 = ( 3.1415926 / 180 ) * camera.fov / 2;  //theta / 2 in radians
   ratio  = camera.w / (double)camera.h;
   a = f * ratio * tan ( theta2 );
   b = a - camera.es / 2.0;
   c = a + camera.es / 2.0;
   N = near;
   F = far;
   T = N * tan ( theta2 );
   B = -T;
   L = -b * N / f;
   R =  c * N / f;
   glFrustum( L, R, B, T, N, F );

   // Create the model for left eye
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   camera.p = Point3( -camera.es/2, 0, f );   //change camera viewpoint
   camera.focus = Point3 ( -camera.es/2, 0, 0 );
   camera.lookAt();
   drawScene();
   glFlush();
   glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE); 

   // Write over the accumulation buffer 
   glAccum(GL_LOAD,1.0); 

   glDrawBuffer(GL_BACK);
   glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

   //now handle the right eye
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   
   // Obtain right-eye parameters from left-eye, no change in T and B
   double temp;
   temp = R;
   R = -L;
   L = -temp;
   glFrustum( L, R, B, T, N, F );
   
   // Right eye filter 
   glColorMask(GL_FALSE,GL_FALSE,GL_TRUE,GL_TRUE);

   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   camera.p = Point3( camera.es/2, 0, f );   //change camera viewpoint
   camera.focus = Point3 ( camera.es/2, 0, 0 );
   camera.lookAt();
   drawScene();
   glFlush();
   glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE); 

   // Add  the new image and copy the result back 
   glAccum(GL_ACCUM,1.0);
   glAccum(GL_RETURN,1.0);
   
   glutSwapBuffers(); 
}

/*
   Create the geometry for a sphere with center at (0, 0, 0)
*/
void drawScene(void)
{
  glPushMatrix();
  glRotatef ( 20, 1, 0, 0 );
  glRotatef ( 45, 0, 1, 0 );
  glLineWidth ( 3 );
  glutWireCube( 4 );
  glPopMatrix();
  glPushMatrix();
  glTranslatef ( 1, 0, 4 );
  glRotatef ( 25, 0, 1, 0 );
  glutSolidTeapot ( 1 );
  glPopMatrix();
}


void keyboard(unsigned char key,int x, int y)
{
   switch (key) {
   case 27:        //ESC  ( quit )
   case 'Q':
   case 'q': 
      exit(0); 
   }
}

int main(int argc,char **argv)
{
   // Set things (glut) up 
   glutInit(&argc,argv);
   glutInitDisplayMode(GLUT_DOUBLE | GLUT_ACCUM | GLUT_RGB | GLUT_DEPTH);

   // Create the window and handlers 
   glutCreateWindow("Two-center Anaglyph Demo");
   glutReshapeWindow( 400, 300 );
   glutInitWindowPosition(100, 100);
   glutDisplayFunc(display);
   glutKeyboardFunc(keyboard);
   init();
   //perpetual loop
   glutMainLoop();
   return(0);
}
