/*
  tests.cpp
  Sample program showing how to write GL shader programs.
  Shader sources are in files "tests.vert" and "tests.frag".
  Fore June
*/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <GL/glew.h>
#include <GL/glut.h>

using namespace std;

/*
   Global handles for the currently active program object, with its two shader objects
*/
GLuint programObject = 0;
GLuint vertexShaderObject = 0;
GLuint fragmentShaderObject = 0;
static GLint win = 0;

int readShaderSource(char *fileName, GLchar **shader )
{
    // Allocate memory to hold the source of our shaders.
    FILE *fp;
    int count, pos, shaderSize;
     
    fp = fopen( fileName, "r");
    if ( !fp )
        return 0;
    struct stat statBuf;
    stat ( fileName, &statBuf );
    shaderSize = statBuf.st_size;    
    if ( shaderSize <= 0 ){
        printf("Shader %s empty\n", fileName);
        return 0;
    }

    *shader = (GLchar *) malloc( shaderSize + 1);
    
    // Read the source code
    count = (int) fread(*shader, 1, shaderSize, fp);
    (*shader)[count] = '\0';

    if (ferror(fp))
        count = 0;

    fclose(fp);
    
    return 1;
}


int installShaders ( const GLchar *vs, const GLchar *fs )
{
    GLint  vertCompiled, fragCompiled;  // status values
    GLint  linked;
    
    // Create a vertex shader object and a fragment shader object
    vertexShaderObject = glCreateShader ( GL_VERTEX_SHADER );
    fragmentShaderObject = glCreateShader ( GL_FRAGMENT_SHADER );
    
    // Load source code strings into shaders, compile and link
    glShaderSource ( vertexShaderObject, 1, &vs, NULL );
    glShaderSource ( fragmentShaderObject, 1, &fs, NULL );

    glCompileShader ( vertexShaderObject );
    glGetShaderiv ( vertexShaderObject, GL_COMPILE_STATUS, &vertCompiled );

    glCompileShader ( fragmentShaderObject );
    glGetShaderiv( fragmentShaderObject, GL_COMPILE_STATUS, &fragCompiled);

    if (!vertCompiled || !fragCompiled)
        return 0;

    // Create a program object and attach the two compiled shaders
    programObject = glCreateProgram();
    glAttachShader( programObject, vertexShaderObject);
    glAttachShader( programObject, fragmentShaderObject);

    // Link the program object 
    glLinkProgram(programObject);
    glGetProgramiv(programObject, GL_LINK_STATUS, &linked);

    if (!linked)
        return 0;

    // Install program object as part of current state
    glUseProgram ( programObject );

    return 1;
}

int init(void)
{
   
   GLchar *VertexShaderSource, *FragmentShaderSource;
   int loadstatus = 0;

   const char *version  = (const char *) glGetString ( GL_VERSION );
   if (version[0] != '2' || version[1] != '.') {
      printf("This program requires OpenGL 2.x, found %s\n", version);
      return 0;
   }
   readShaderSource("tests.vert", &VertexShaderSource );
   readShaderSource("tests.frag", &FragmentShaderSource );
   loadstatus = installShaders(VertexShaderSource, FragmentShaderSource);
   if ( !loadstatus ) {
     printf("\nCompilation of shaders not successful!\n");
   }  
 
   return loadstatus;
}

static void reshape(int width, int height)
{
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   glFrustum(-1.0, 1.0, -1.0, 1.0, 5.0, 25.0);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   glTranslatef(0.0f, 0.0f, -15.0f);
}

void CleanUp(void)
{
   glDeleteShader(vertexShaderObject);
   glDeleteShader(fragmentShaderObject);
   glDeleteProgram(programObject);
   glutDestroyWindow(win);
}

static void key(unsigned char key, int x, int y)
{
   switch(key) {
   case 27:
      CleanUp();
      exit(0);
      break;
   }
   glutPostRedisplay();
}

void display(void)
{
   glClearColor ( 1.0, 1.0, 1.0, 0.0 );	//get white background color
   glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
   glColor3f ( 0, 1, 0 );  		//green, this will have no effect if shader is loaded
   glLineWidth ( 4 );
   glutWireSphere ( 2.0, 16, 8 );
   glutSwapBuffers();
   glFlush();
}

int main(int argc, char *argv[])
{
   glutInit(&argc, argv);
   glutInitWindowPosition( 0, 0 );
   glutInitWindowSize ( 300, 300 );
   glutInitDisplayMode ( GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH );
   win = glutCreateWindow(argv[0]);
   glutReshapeFunc ( reshape );
   glutDisplayFunc ( display );
   glutKeyboardFunc ( key );
   // Initialize the "OpenGL Extension Wrangler" library
   glewInit();

   int successful = init();
   if ( successful )
     glutMainLoop();

   return 0;
}

