/* ppmdemo.cpp
 * Demostrate read and write of PPM files.
 * Compile: g++ -o ppmdemo ppmdemo.cpp 
 * Execute: ./ppmdemo
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

using namespace std;

//A public class is the same as a 'struct'
class CImage {
public:
  unsigned char red;
  unsigned char green;
  unsigned char blue;
};

/*
 * Create PPM header from image width and height.  "P6" format is used.
 * PPM header returned in integer array ppmh[].
 */ 
void make_ppm_header ( int ppmh[], int width, int height )
{
  //standard header data, 'P' = 0x50, '6' = 0x36, '\n' = 0x0A
  int ca[] = {0x50, 0x36, 0x0A, 				//"P6"
	      0x33, 0x36, 0x30, 0x20, 0x32, 0x38, 0x38, 0x0A,   //image width=260, height = 288
	      0x32, 0x35, 0x35, 0x0A };				//color levels / pixel = 256

  //only have to change width and height
  char temp[10], k;

  sprintf(temp, "%3d", width );			//width in ascii code
  k = 0;
  for ( int i = 3; i <= 5; ++i )		//replace width
    ca[i] = temp[k++];

  sprintf(temp, "%3d", height );		//height in ascii code
  k = 0;
  for ( int i = 7; i <=9; ++i )			//replace height
    ca[i] = temp[k++];

  for ( int i = 0; i < 15; ++i )		//form header
    ppmh[i] = ca[i];
}

void save_ppmdata ( FILE *fp, CImage *ip, int width, int height )
{
  int size = width * height;

  for ( int i = 0; i < size; ++i ){
    putc ( ip[i].red, fp );
    putc ( ip[i].green, fp );
    putc ( ip[i].blue, fp );
  }
}

void ppm_read_comments ( FILE *fp )
{
  int c;
  while ( (c = getc ( fp ) )  == '#' ) {
    while (  getc( fp )  != '\n' )
	;
  }
  ungetc ( c, fp );
}  

class ppm_error
{
  public:
    ppm_error() {
      printf("\nIncorrect PPM format!\n");
      exit ( 1 );
    }
};

int main()
{
  int ppmh[20], c;				     //PPM header
  int width = 32, height = 32;			     //image width and height for demo purpose  
  make_ppm_header ( ppmh, width, height );
  FILE *input = fopen ("../data/testread.ppm", "rb");//PPM file for testing read
  FILE *output = fopen ("testwrite.ppm", "wb");	     //PPM file for testing write

  if ( input == NULL ) {
    printf("\nError opening file ../data/testread.ppm!\n");
    return 1;
  }
  if ( output == NULL ) {
    printf("\nError opening file testwrite.ppm!\n");
    return 1;
  } 
  
  //write demo
  for ( int i = 0; i < 15; ++i )		//save PPM header
    putc ( ppmh[i], output );

  CImage image[width][height];
  for ( int i = 0; i < height; ++i ) {		//create a red rectangle
    for ( int j = 0; j < width; ++j ) {
	image[i][j].red = 255;			//red component
	image[i][j].green = 0;			//green component	
	image[i][j].blue = 0;			//blue component
    }
  }

  save_ppmdata ( output, (CImage*) image, width, height );
  printf("\nPPM file testwrite.ppm created!\n");
  fclose ( output );

  //read demo
  ppm_read_comments ( input );			//read comments
  char temp[100];
  fscanf ( input, "%2s", temp );
  temp[3] = 0;
  if ( strncmp ( temp, "P6", 2 ) ) 
    throw ppm_error();
  ppm_read_comments ( input );
  fscanf ( input, "%d", &width );
  ppm_read_comments ( input );
  fscanf ( input, "%d", &height );
  ppm_read_comments ( input );
  int colorlevels;
  fscanf ( input, "%d", &colorlevels );
  printf("\n%s PPM file: ", temp );
  printf(" \n\twidth=%d\theight=%d\tcolorlevles=%d\n", width,height,colorlevels+1 ); 
  ppm_read_comments ( input );
  while ( ( c = getc ( input )) == '\n' );	//get rid of extra line returns
  ungetc ( c ,input ); 

  //save the data in another file
  CImage ibuf[width][height];
  fread ( ibuf,  3, width * height, input );
  output = fopen ( "test.ppm", "wb" );  	//save PPM data in new file "test.ppm"
  make_ppm_header ( ppmh, width, height );
  for ( int i = 0; i < 15; ++i )		//save PPM header
    putc ( ppmh[i], output );
  save_ppmdata ( output, (CImage*) ibuf, width, height );	//save data
  printf("\nPPM file test.ppm created!\n");

  fclose ( input );
  fclose ( output );
  return 0;
}
