/*
 * stereo.cpp
 * Render stereo pairs.  Scene defined in drawScene().
 * Fore June
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <GL/glut.h>
#include <algorithm>
#include <iostream>
#include <vector>
#include "Point3.h"
#include "Vector3.h"
#include "Camera.h"

using namespace std;

void threePawns();
void seaShell();
void aVase();

const double PI = 3.14159265389;
int anglex= 8, angley = 16, anglez = 0; 
char showType = 's';  //showing seashell, pawns or vase

Camera camera;

/*
   Initialization.
   This is where global OpenGL/GLUT settings are made. 
*/
void init(void)
{
   glEnable(GL_DEPTH_TEST);
   glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
   glFrontFace(GL_CW);
   glClearColor(0.0, 0.0, 0.0, 0.0);
   glClearAccum(0,0,0,0.0);   // The default 
   glEnable( GL_CULL_FACE );
   glCullFace ( GL_BACK );
   glPolygonMode( GL_FRONT, GL_FILL );
   glShadeModel(GL_SMOOTH);

   //lighting
   GLfloat mat_specular[] = { 1.0, 1.0, 1.0, 1.0 };
   GLfloat mat_shininess[] = { 50.0 };
   GLfloat light[] = { 1.0, 1.0, 1.0 };
   GLfloat light1[] = { 1.0, 1, 1 };
   GLfloat light_position[] = {0, 1.0, 1.0, 0.0 };
   GLfloat light_position1[] = { -1.0, -1.0, -1.0, 0.0 };
   GLfloat lmodel_ambient[] = { 0.8, 0.8, 0.8, 1.0 };

   glShadeModel (GL_SMOOTH);
   glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
   glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
   glLightfv(GL_LIGHT0, GL_POSITION, light_position);
   glLightfv(GL_LIGHT1, GL_POSITION, light_position1);

   glLightfv(GL_LIGHT0, GL_DIFFUSE, light );
   glLightfv(GL_LIGHT0, GL_SPECULAR, light );
   glLightfv(GL_LIGHT1, GL_DIFFUSE, light1 );
   glLightfv(GL_LIGHT1, GL_SPECULAR, light1 );
   glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);

   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   glEnable(GL_LIGHT1);
   glEnable(GL_DEPTH_TEST);
   printf("\nPress 'p' to display 3 pawns, \n      's' to display seashell,");
   printf("\n      'v' to display vase,    \n      'x', 'y', 'z' to rotate scene.\n");
}

void rotate()
{
   glRotatef( anglex, 1.0, 0.0, 0.0);   //rotate the object about x-axis
   glRotatef( angley, 0.0, 1.0, 0.0);   //rotate about y-axis   
   glRotatef( anglez, 0.0, 0.0, 1.0);   //rotate about z-axis
}

void drawScene(void)
{
  glPushMatrix();
  rotate();
  if ( showType == 'p' )
    threePawns();
  else if ( showType == 's' )
    seaShell();
  else
    aVase();

  glPopMatrix();
}


/*
   This is the basic display callback routine.
   It creates the geometry, viewing position, and focus.
   It renders left eye first, then right.
*/
void display(void)
{
   glClearAccum ( 0, 0, 0, 0 );   // The default 
   glClear( GL_DEPTH_BUFFER_BIT );
   /*
     L = left, R = right, T = top, B = bottom, N = near, F = far, f = focal length
     e = eye separation, a = half-width of projection plane, ratio = aspect ratio
     theta2 = (field of view)/2
   */
   double theta2, near,far;
   double L, R, T, B, N, F, f, e, a, b, c, ratio;
   glClearColor(0, 0, 0, 0.0);     //black background
   glClear ( GL_COLOR_BUFFER_BIT );

   // Clip to avoid extreme stereo 
   near = camera.f / 5;
   //near = camera.f / 4;
   far = 1000.0;
   f = camera.f;

   // Set the buffer for writing and reading 
   glDrawBuffer ( GL_BACK );
   glReadBuffer ( GL_BACK );

   // Clear things 
   glClear(GL_ACCUM_BUFFER_BIT); 

   // Left eye filter (red(
   glColorMask ( GL_TRUE, GL_FALSE, GL_FALSE, GL_TRUE );

   // Create the projection 
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   theta2 = ( 3.1415926 / 180 ) * camera.fov / 2;  //theta / 2 in radians
   ratio  = camera.w / (double)camera.h;
   a = f * ratio * tan ( theta2 );
   b = a - camera.es / 2.0;
   c = a + camera.es / 2.0;
   N = near;
   F = far;
   T = N * tan ( theta2 );
   B = -T;
   L = -b * N / f;
   R =  c * N / f;
   glFrustum( L, R, B, T, N, F );
   // Create the model for left eye
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   camera.p = Point3( -camera.es/2, 0, f );   //change camera viewpoint
   camera.focus = Point3 ( -camera.es/2, 0, 0 );
   camera.lookAt();

   glDrawBuffer ( GL_BACK );
   glReadBuffer ( GL_BACK );
   glClearColor(0, 0, 0, 0.0);     //black background
   glClear ( GL_COLOR_BUFFER_BIT );
   drawScene();
   glFlush();

   // Write over the accumulation buffer 
   glAccum ( GL_LOAD, 1.0 ); 

   glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

   //now handle the right eye
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   
   // Obtain right-eye parameters from left-eye, no change in T and B
   double temp;
   temp = R;
   R = -L;
   L = -temp;
   glFrustum( L, R, B, T, N, F );
   
   // Right eye filter  (blue)
   glColorMask ( GL_FALSE, GL_FALSE, GL_TRUE, GL_TRUE );

   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
   camera.p = Point3( camera.es/2, 0, f );   //change camera viewpoint
   camera.focus = Point3 ( camera.es/2, 0, 0 );
   camera.lookAt();

   glClear ( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

   drawScene();
   glFlush();

   // Add  the new image 
   glAccum ( GL_ACCUM, 1.0 );

   // Allow all color components
   glColorMask ( GL_TRUE, GL_TRUE, GL_TRUE,GL_TRUE ); 
   // Copy result back
   glAccum ( GL_RETURN, 1.0 );
   glFlush();
   glutSwapBuffers(); 
}

void keyboard(unsigned char key,int x, int y)
{
   switch (key) {
   case 27:        //ESC  ( quit )
   case 'Q':
   case 'q': 
      exit(0); 
    case 'x':
      anglex = ( anglex + 3 ) % 360;
      break;
    case 'X':
      anglex = ( anglex - 3 ) % 360;
      break;
    case 'y':
      angley = ( angley + 3 ) % 360;
      break;
    case 'Y':
      angley = ( angley - 3 ) % 360;
      break;
    case 'z':
      anglez = ( anglez + 3 ) % 360;
      break;
    case 'Z':
      anglez = ( anglez - 3 ) % 360;
      break;
    case 'r':                                   //reset
      anglez = angley = anglex = 0;
      glLoadIdentity();
      break;
    case 'p':
      showType = 'p';
      anglex = angley = anglez = 0;
      break;
    case 's':
      showType = 's';
      anglex = 9; angley = 16;  anglez = 0;
      break;
    case 'v' :
      showType = 'v';
      anglex = angley = anglez = 0;
      break;
   }
   glutPostRedisplay();
}

int main(int argc,char **argv)
{
   // Set things (glut) up 
   glutInit(&argc,argv);
   glutInitDisplayMode( GLUT_DOUBLE | GLUT_ACCUM | GLUT_RGB | GLUT_DEPTH );

   // Create the window and handlers 
   glutCreateWindow("Seashell, Pawns, Vase Anaglyph");
   glutReshapeWindow( 800, 600 );
   glutInitWindowPosition(100, 100);
   glutDisplayFunc(display);
   glutKeyboardFunc(keyboard);
   init();
   //perpetual loop
   glutMainLoop();
   return(0);
}
