/*
 * pawns.cpp
 * Create pawns using surface of revolution with B-splines.
 * Fore June
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <GL/glut.h>
#include <iostream>
#include "Vector3.h"   //in directory ../12

using namespace std;

static const double PI = 3.14159265389;
const int Npoints = 27;
const int m_order = 4;


GLfloat ctrlpoints[Npoints][3] = {
{4.38,0.00, 0},{4.22,0.50, 0},{3.98,0.72, 0},{3.62,0.80, 0},{3.24,0.66, 0},
{3.10,0.56, 0},{2.92,0.80, 0},{2.76,0.88, 0},{2.58,0.74, 0},{2.48,0.44, 0},
{2.10,0.48, 0},{1.74,0.60, 0},{1.54,0.66, 0},{1.44,0.90, 0},{1.10,1.04, 0},
{0.90,1.18, 0},{0.76,1.38, 0},{0.68,1.48, 0},{0.62,1.48, 0},{0.58,1.44, 0},
{0.52,1.36, 0},{0.44,1.38, 0},{0.30,1.46, 0},{0.28,1.54, 0},{0.22,1.50, 0},
{0.10,1.50, 0},{0.00,1.50, 0}
};

/*
 *   Build standard knot vector for n control points
 *   and B-splines of order m. U will hold the knot vector.
 */
void setKnotVector ( int m, int n, float U[] )
{
  if ( n < m  ) return;         //not enough control points
  for ( int i = 0; i < n + m; ++i ){
    if (i < m) U[i] = 0.0;
    else if (i < n) U[i] = i - m + 1;        //i is at least m here
    else U[i] = n - m + 1;
  }
}

//evaluate order m blending functions recurvsively, U[] holds knots
float N_k ( int k, int m, float u, float U[] )
{
  float d1, d2, sum = 0.0;

  if ( m == 1 )
    return ( U[k] < u &&  u <= U[k+1] );   //1 or 0
   d1 = U[k+m-1] - U[k];
  if ( d1 != 0 )
    sum = (u - U[k]) * N_k ( k, m-1, u, U ) / d1;
  d2 = U[k+m] - U[k+1];
  if ( d2 != 0 )
    sum += (U[k+m] - u) * N_k ( k+1, m-1, u, U ) / d2;

  return sum;
}

//non uniform rational B-splines, n control points, order m, p[] is the output point
void nurb ( float control_points[][3], float u, float U[], float p[] )
{
  // sum the control points mulitplied by their respective blending functions
   for ( int i = 0; i < 3; ++i ){        //x, y, z components
    p[i] = 0;
    for ( int k = 0; k < Npoints; ++k )
      p[i] += N_k ( k, m_order, u, U ) * control_points[k][i];
  }
}

//construct pawn using surface of revolution
void pawn (int nx, int ntheta, float startx, float endx )
{
   const int n = Npoints, m = m_order;  //n control points, degree m NURB
   float U[n+m];

   const float dx = (endx - startx)/nx; //x step size
   const float dtheta = 2*PI / ntheta;  //angular step size
   float theta = PI/2.0;                //from pi/2 to3pi/2
   setKnotVector(m, n, U);

   int i, j;
   float x, y, z, r;                    //current coordinates
   float x1, y1, z1, r1;                //next coordinates
   float t, v[3];
   double  va[3], vb[3], vc[3], normal[3];
   int nturn = 0;
   x = startx;
   nurb ( ctrlpoints, 0, U, v );
   x = v[0];
   r = v[1];
   bool first_point = true;
   for ( int k = m-1; k < n; ++k ) {    //step through the knots
      float dknot = U[k+1] - U[k];
      if ( dknot == 0 ) continue;

      theta = 0;        //PI / 2.0;
      int start=0, nn=60, end=nn;
      for (i = start; i <= end; i++) {
       t = U[k] + dknot * (float) i /nn;
       nurb (ctrlpoints, t, U, v);
       if ( first_point ) {
          v[0] = ctrlpoints[0][0];
          v[1] = ctrlpoints[0][1];
          first_point = false;
       }
       x1 =  v[0];
       r1 = v[1];

       //draw the surface composed of quadrilaterals by sweeping theta
       glBegin( GL_QUAD_STRIP );
        for ( j = 0; j <= ntheta; ++j ) {
          theta += dtheta;
          double cosa = cos( theta );
          double sina = sin ( theta );
          y = r * cosa;  y1 = r1 * cosa;        //current and next y 
          z = r * sina;  z1 = r1 * sina;        //current and next z
          if ( nturn == 0 ) {
            va[0] = x;  va[1] = y;      va[2] = z;
            vb[0] = x1; vb[1] = y1;     vb[2] = z1;
            nturn++;
          } else {
            nturn = 0;
            vc[0] = x;  vc[1] = y;      vc[2] = z;
            plane_normal ( normal, va, vb, vc );
            glNormal3f ( normal[0], normal[1], normal[2] );
          }
          glVertex3f (x, y, z);
          glVertex3f (x1, y1, z1);
        }
      glEnd();
      x = x1;
      r = r1;
     }
   } //for k 
}

// Draw 3 pawns on a board
void threePawns(void)
{
   int thePawn = glGenLists (1);
   glNewList( thePawn, GL_COMPILE);
     pawn(32, 64, 0, 3.5);
   glEndList();

    glPushMatrix();
    //Make better orientation
    glTranslatef ( 0, -1, 0 );
    glRotatef( 90, 0, 0, 1 );
    glRotatef( 10, 1, 0, 0 );

    glPushMatrix();
    //Turn off light when drawing a board
    glTranslatef( -0.1, 0, 0 );
    glScalef ( 0.1, 6, 6 );
    glDisable ( GL_LIGHTING );
    glColor3f ( 0.8, 0.8, 0.8 );
    glutSolidCube ( 2 );
    glColor3f ( 1, 1, 1 );
    //Draw outline of board
    glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    glutWireCube ( 2 );
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    glEnable ( GL_LIGHTING );
    glPopMatrix();

    //Now draw 3 pawns at various positions
    glPushMatrix();
    glTranslatef ( 0, 0.5, -3 );
    glCallList ( thePawn );
    glPopMatrix();

    glPushMatrix();
    glTranslatef ( 0, -3, 0.5 );
    glCallList ( thePawn );
    glPopMatrix();

    glPushMatrix();
    glTranslatef ( 0, 3, 0.5 );
    glCallList ( thePawn );
    glPopMatrix();

    glPopMatrix();
}

