/*  nurbsurface.cpp
 *  This program demonstrates the use of NURBS to generate a surface.
 *   Fore June
 */
#include <GL/glut.h>
#include <stdlib.h>
#include <stdio.h>

using namespace std;

const int uOrder = 4;                                                                  
const int vOrder = 4;
                                                                                
float controlPoints[uOrder][vOrder][3] = {
   {{-2, -1, 4.0}, {-1, -1, 3.0},
    {0, -1, -1.5}, {2, -1, 2.5}},
   {{-2, -0.5, 1.0}, {-1, -0.5, 3.0},
    {0, -0.5, 0.0}, {2, -0.5, -1.0}},
   {{-2, 0.5, 4.0}, {-1, 0.5, 1.0},
    {0, 0.5, 2.0}, {2, 0.5, 4.0}},
   {{-2, 1.5, -2.0}, {-1, 1.5, -2.0},
    {0, 1.5, 0.0}, {2, 1.5, -1.0}}
};

GLUnurbsObj *nurbs;
          
void nurbsError( GLenum error )
{
   const GLubyte *s = gluErrorString ( error );
   printf ( "Nurbs Error: %s\n", s );
   exit (0);
}
                                                                     
void init(void)
{
   glClearColor(1.0, 1.0, 1.0, 0.0);
   GLfloat mat_ambient[] = { 0.5, 0.4, 0.3, 1.0 };
   GLfloat mat_diffuse[] = { 0.7, 0.8, 0.9, 1.0 };
   GLfloat mat_specular[] = { 0.9, 0.8, 0.7, 1.0 };
   GLfloat mat_shininess[] = { 50.0 };

   glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
   glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
   glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);
   glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
   GLfloat light[] = { 1, 1, 1 };
   GLfloat light_position[] = { 1.0, 1.0, 1.0, 0.0 };
   glLightfv(GL_LIGHT0, GL_DIFFUSE, light );
   glLightfv(GL_LIGHT0, GL_AMBIENT, light );
   glLightfv(GL_LIGHT0, GL_SPECULAR, light );
   glLightfv(GL_LIGHT0, GL_POSITION, light_position);

   glEnable(GL_LIGHTING);
   glEnable(GL_LIGHT0);
   glEnable(GL_DEPTH_TEST);
   glEnable(GL_AUTO_NORMAL);
   glEnable(GL_NORMALIZE);

   nurbs = gluNewNurbsRenderer();
   gluNurbsProperty( nurbs, GLU_SAMPLING_TOLERANCE, 25.0);
   gluNurbsProperty( nurbs, GLU_DISPLAY_MODE, GLU_FILL);
   gluNurbsCallback( nurbs, GLU_ERROR, (GLvoid (*)()) nurbsError);
}
                                                                                
void display(void)
{
   glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
   gluLookAt ( 10, 10, 10, 0, 0.0, 0.0, 0.0, 1.0, 0.0);
   glEnable ( GL_LINE_SMOOTH );
   glLineWidth( 2 ); 

   const int uKnotCount = 8, vKnotCount = 8;  
   float uKnots[uKnotCount] = {0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0};
   float vKnots[vKnotCount] = {0.0, 0.0, 0.0, 0.0, 1.0, 1.0, 1.0, 1.0};
   int uStride = vOrder * 3, vStride = 3;

   gluBeginSurface ( nurbs );
   gluNurbsSurface ( nurbs, 
                     uKnotCount, uKnots, vKnotCount, vKnots,
                     uStride, vStride, &controlPoints[0][0][0], 
                     uOrder, vOrder, GL_MAP2_VERTEX_3);
   gluEndSurface ( nurbs );
/*
   // The following code displays the control points as dots. 
   glPointSize( 6.0 );
   glColor3f( 1.0, 0.0, 0.0 );
   glEnable ( GL_POINT_SMOOTH );
   glBegin(GL_POINTS);
      for ( i = 0; i < uOrder; i++ )
	for ( j = 0; j < vOrder; j++ )
         glVertex3fv( &controlPoints[i][j][0] );
   glEnd();
*/
   glFlush();
}
 
void reshape(int w, int h)
{
   glViewport(0, 0, (GLsizei) w, (GLsizei) h);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   if (w <= h)
      glOrtho(-5.0, 5.0, -5.0*(float)h/(float)w,
               5.0*(float)h/(float)w, 5.0, 25.0);
   else
      glOrtho(-5.0*(float)w/(float)h,
               5.0*(float)w/(float)h, -5.0, 5.0, 5.0, 25.0);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}
                                                                                
void keyboard(unsigned char key, int x, int y)
{
   switch (key) {
      case 27:
         exit(0);
         break;
   }
}
                                                                                
int main(int argc, char** argv)
{
   glutInit(&argc, argv);
   glutInitDisplayMode (GLUT_SINGLE | GLUT_RGB);
   glutInitWindowSize (500, 500);
   glutInitWindowPosition (100, 100);
   glutCreateWindow (argv[0]);
   init ();
   glutDisplayFunc(display);
   glutReshapeFunc(reshape);
   glutKeyboardFunc (keyboard);
   glutMainLoop();
   return 0;
}
