/*
  bspline.cpp
  Sample implementations of setting NURB knot vectors, B-Spline 
  blending (basis) functions and usage of B-Splines.
  Fore June
*/
#include <GL/glut.h>
#include <stdlib.h>
#include <stdio.h>
#include "Point3.h"

using namespace std;

const int Npoints = 8;    //Number of control points
const int m_order = 4;    //order of blending function

//sample control points ( world window is x: -5 -> 5; y: -5 -> 5)
double ctrlpoints[Npoints][3] = {
   { -4, 0, 0 }, { -3, 2.5, 0}, {-2, 2, 0}, { 0, 1, 0},
   { 1.5, 0, 0}, { 2, -1, 0}, {3, 0.5,  0}, {4, 1, 0}}; 

Point3 control_points[Npoints];

/*
 * Build standard knot vector for n control points
 * and B-splines of order m. U will hold the knot vector.
 */
void setKnotVector ( int m, int n, float U[] )
{
  if ( n < m  ) return;              //not enough control points
  for ( int i = 0; i < n + m; ++i ){
    if (i < m) U[i] = 0.0;
    else if (i < n) U[i] = i - m + 1;//i is at least m here
    else U[i] = n - m + 1;
  }
}

//evaluate order m blending functions recurvsively, U[] holds knots
float N_k ( int k, int m, float u, float U[] )
{
  float d1, d2, sum = 0.0;

  if ( m == 1 )
    return ( U[k] < u &&  u <= U[k+1] );   //1 or 0

  //m larger than 1, so evaluate recursively
  d1 = U[k+m-1] - U[k];
  if ( d1 != 0 )
    sum = (u - U[k]) * N_k(k,m-1,u, U) / d1;
  d2 = U[k+m] - U[k+1];
  if ( d2 != 0 )
    sum += ( U[k+m] - u) * N_k(k+1, m-1, u, U ) / d2;

  return sum;
}

//non uniform rational B-splines, n control points, order m; p is the output point 
void nurb ( int n, int m, const Point3 control_points[], float u, float U[],  Point3 &p )
{
  // sum the control points mulitplied by their respective blending functions
  Point3 p3;           //x, y, z components set to zero
  for ( int k = 0; k < n; ++k ){
      float Nk = N_k(k, m, u, U);
      p3.x += Nk * control_points[k].x;
      p3.y += Nk * control_points[k].y;
      p3.z += Nk * control_points[k].z;
  }
  p = p3;              //= is a copy command in C++
}

void init(void)
{
   glClearColor(1.0, 1.0, 1.0, 0.0);
   glShadeModel(GL_FLAT);
   glEnable ( GL_POINT_SMOOTH ); 
   glEnable ( GL_LINE_SMOOTH ); 
   for ( int i = 0; i < Npoints; i++ ) {
      Point3 p( ctrlpoints[i][0], ctrlpoints[i][1], ctrlpoints[i][2] );
      control_points[i] = p;
   }
}

void testing()
{
  int n = Npoints, m = m_order; //n control points, order m
  // printf("\nEnter number of control points: ");
  //  scanf("%d", &n );
  float U[n+m];

  setKnotVector ( m,  n, U );

  for ( int i = 0; i < n + m; i++ )
    printf("%4.2f,", U[i] );
}

void display(void)
{
   int i;
   float x, y, t;
   Point3 p3;       //a Point3 object
   glClear ( GL_COLOR_BUFFER_BIT );
   glColor3f ( 0, 0, 0 );
   glPointSize ( 6.0 );
   const int n = Npoints, m = m_order;  //this reduces to cubic B-spline if n = 4
   float U[n+m];                        //knot vector
   setKnotVector ( m,  n, U );          //construct the knot vector
   bool first_point = true;
   //evaluate the curve using nurb
   glBegin(GL_LINE_STRIP);
   for ( int k = m-1; k < n; ++k ) {
      float dknot = U[k+1] - U[k];
      if ( dknot == 0 ) continue;
      int start=0, end=30;
      if ( k == n - 1 ) end = 30;
      for (i = start; i <= end; i++) {
         t = U[k] + dknot * (float) i /30.0;  
         nurb ( Npoints, m_order, control_points,  t, U, p3 );
         if ( first_point ) {
           p3.x = control_points[0].x;
           p3.y = control_points[0].y;
           first_point = false;
         }
         glVertex2f ( p3.x, p3.y );
      }
   } //for k
   glEnd();

   /* The following code displays the control points as dots. */
   glColor3f( 1.0, 0.0, 0.0);
   glBegin(GL_POINTS);
      for (i = 0; i < Npoints; i++){
         glVertex3f(control_points[i].x,control_points[i].y, control_points[i].z);
      }
   glEnd();

   glFlush();
}

void reshape(int w, int h)
{
   glViewport(0, 0, (GLsizei) w, (GLsizei) h);
   glMatrixMode(GL_PROJECTION);
   glLoadIdentity();
   if (w <= h)
      glOrtho(-5.0, 5.0, -5.0*(GLfloat)h/(GLfloat)w,
               5.0*(GLfloat)h/(GLfloat)w, -5.0, 5.0);
   else
      glOrtho(-5.0*(GLfloat)w/(GLfloat)h,
               5.0*(GLfloat)w/(GLfloat)h, -5.0, 5.0, -5.0, 5.0);
   glMatrixMode(GL_MODELVIEW);
   glLoadIdentity();
}

void keyboard(unsigned char key, int x, int y)
{
   switch (key) {
      case 27:
         exit(0);
         break;
   }
}

int main(int argc, char** argv)
{
   glutInit(&argc, argv);
   glutInitDisplayMode (GLUT_SINGLE | GLUT_RGB);
   glutInitWindowSize (500, 500);
   glutInitWindowPosition (100, 100);
   glutCreateWindow (argv[0]);
   init ();
   glutDisplayFunc(display);
   glutReshapeFunc(reshape);
   glutKeyboardFunc (keyboard);
   glutMainLoop();

   return 0;
}
