/**
 * treesearch.cpp
 *   Search for a certain node.
 *   Demonstrate different ways of extracting the subtree of the node.
 *   Most likely, your system already has libxml2 installed.
 *   If not, download it from http://xmlsoft.org and install it.
 */
#include <stdio.h>
#include <string.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xmlreader.h>

void parseNode ( xmlDocPtr doc, xmlNodePtr cur )
{
    xmlChar *key;
    cur = cur->xmlChildrenNode;
    while (cur != NULL) {
        key = xmlNodeListGetString(doc, cur, 1);
	printf(" %s\n", key);
	xmlFree(key);
        cur = cur->next;
    } 
    return;
}

void parseParam( xmlDocPtr doc, xmlNodePtr cur )
{
    xmlChar *key;
    if ( cur != NULL ) {
      printf(": %s\n", cur->name );
        key = xmlNodeListGetString(doc, cur, 1);
	printf(" %s\n", key);
	xmlFree(key);
   }
}


void print_element_names(xmlNode *root, int level )
{
    xmlNode *cur_node = NULL;
    xmlChar *key;
    ++level;   //one level deeper in next call

    for (cur_node = root; cur_node; cur_node = cur_node->next) {
        if (cur_node->type == XML_ELEMENT_NODE) {
          for ( int i = 0; i < level; i++ )
            printf(" +");  //signifies level of node
          printf(" %s\n", cur_node->name );
        }
        print_element_names( cur_node->children, level );
    }
}


//Search descendent of node, using given name
xmlNode *searchNode ( xmlNode *a_node, char target[] )
{
  xmlNode *nodeFound = NULL;

  for ( xmlNode *cur = a_node; cur; cur= cur->next) {
    if (cur->type == XML_ELEMENT_NODE) {
      if ( !xmlStrcmp (  cur->name, (const xmlChar* )target ) ) {
         printf("Found %s \n", cur->name );
	 nodeFound = cur;
	 break;
       }
     }
     //search recursively until node is found.
     if ( nodeFound == NULL && cur != NULL ) 
        nodeFound = searchNode ( cur->children, target );
  }
  
  return nodeFound;
}
/**
 * Simple example to parse a collada file called "cube.dae", 
 * Search for a node, print out the subtree node names.
 * Extract contents of nodes.
 * Demo different ways of using xmlElemDump() to extract information.
 * Nodes to search in the example are: source, float_array, library_geometries, vertices  
 */
int main(int argc, char **argv)
{
    xmlDoc *doc = NULL;
    xmlNode *root_element = NULL;
    const char filename[] = "cube.dae";

    // Initialize library and check whether correct version is used.
    LIBXML_TEST_VERSION

    // parse the file and get the DOM 
    doc = xmlReadFile( filename, NULL, 0);

    if ( doc == NULL ) {
        printf( "error: could not parse file %s\n", filename );
    }

    // Get the root element node 
    root_element = xmlDocGetRootElement( doc );
  
    xmlNode *nodeFound;

    nodeFound = searchNode ( root_element, "source" );
    //print subtrees rooted at "source"
    print_element_names(nodeFound, -1);

    //find "float_array" within first "source" subtree
    if ( nodeFound != NULL ){
       nodeFound = searchNode ( nodeFound, "float_array" );
    }
    //print out data of node found
    if ( nodeFound != NULL )
      parseNode ( doc, nodeFound ); 

    printf("---------------------------------------------------------\n");
   
    nodeFound = searchNode ( root_element, "library_geometries" );    
    nodeFound = searchNode ( nodeFound, "vertices" );
    if ( nodeFound == NULL ) {
      printf("\nvertices Node not found!\n");
      return 1;
    }
    //send information to screen
    xmlElemDump ( stdout, doc, nodeFound );
    printf("\nparsing the node:\n");
    //open a pipe, which will execute the awk command when writing 
    FILE *fp = popen ( "awk 'BEGIN {} {for(i=NF;i > 0;i--) printf(\"%s\\n\",$i); } END {} '", "w" );
    //print the parsed text to screen
    xmlElemDump ( fp, doc, nodeFound );
    pclose ( fp );  //close the pipe
    printf("---------------------------------------------------------\n");

    //search a new node
    nodeFound = searchNode ( nodeFound, "polylist" );
    if ( nodeFound == NULL ) return 1;
    //open a temporary file
    FILE *f = fopen ("temp.$$$", "w" );
    const int bufsize = 20000;
    char buf[bufsize];       //create a buffer
    bzero ( buf, bufsize );  //set buffer to zeros
    //send the output stream to buf before writing to "temp.$$$"
    setvbuf ( f, buf, _IOFBF, bufsize );

    //dump the element of node to the buffer
    xmlElemDump ( f, doc, nodeFound );
    printf("%s\n", buf ); //print content of node  
 
    /*free the document */
    xmlFreeDoc(doc);
    
    xmlCleanupParser();

    return 0;
}
