package opengl.glslanimate;

import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;

import android.content.Context;
import java.io.IOException;
import android.util.Log;
import android.opengl.GLES20;

public class Triangle {
	
  private static String LOG_APP_TAG = "io_tag";	
  private Context context;
  
  private String vsCode = null;
  private String fsCode = null;		
  private int program;
  private int vertexShader;
  private int fragmentShader;
  private FloatBuffer vertexBuffer;
  private int vertexCount = 3;

   // number of coordinates per vertex in this array
   static final int COORDS_PER_VERTEX = 3;
   static float triangleCoords[] = {   // in counterclockwise order:
	  0.0f,  0.9f, 0.0f, // top vertex
	  -0.5f,  0.1f, 0.0f, // bottom left
	   0.5f,  0.1f, 0.0f  // bottom right
    };

	// Set color with red, green, blue and alpha (opacity) values
	float color[] = { 0.9f, 0.1f, 0.9f, 1.0f };

	// Constructor   
    Triangle( Context context0){
      context = context0;
      // get shader codes from res/raw/vshader and res/raw/fshader
      vsCode = getShaderCode( GLES20.GL_VERTEX_SHADER );
      fsCode = getShaderCode( GLES20.GL_FRAGMENT_SHADER );
      program = GLES20.glCreateProgram(); // create empty OpenGL ES Program
	  
      vertexShader = loadShader(GLES20.GL_VERTEX_SHADER, vsCode );
      fragmentShader = loadShader(GLES20.GL_FRAGMENT_SHADER, fsCode );
      GLES20.glAttachShader ( program, vertexShader ); // add the vertex shader to program    
      GLES20.glAttachShader(program, fragmentShader); // add the fragment shader to program
      GLES20.glLinkProgram(program);                  // creates OpenGL ES program executables
      GLES20.glUseProgram( program);

      // initialize vertex byte buffer for shape coordinates
      ByteBuffer bb = ByteBuffer.allocateDirect(
         // (number of coordinate values * 4 bytes per float)
         triangleCoords.length * 4);
         // use the device hardware's native byte order
         bb.order(ByteOrder.nativeOrder());

         // create a floating point buffer from the ByteBuffer
         vertexBuffer = bb.asFloatBuffer();
         // add the coordinates to the FloatBuffer
         vertexBuffer.put(triangleCoords);
         // set the buffer to read the first coordinate
         vertexBuffer.position(0);
	} //Triangle Constructor
	   
    // Get source code of a shader
	protected String getShaderCode( int type )
    {
		 InputStream inputStream = null;
		 String str = null;
		 try {
			 if ( type == GLES20.GL_VERTEX_SHADER )
		        inputStream = context.getResources().openRawResource(R.raw.vshader);
			 else
			    inputStream = context.getResources().openRawResource(R.raw.fshader);
		        byte[] reader = new byte[inputStream.available()];;
		        while (inputStream.read(reader) != -1) {}
		        str = new String ( reader );
		 }  catch(IOException e) {
             Log.e(LOG_APP_TAG, e.getMessage());
         }
		 
       return  str;
    }

    public static int loadShader (int type, String shaderCode ) {

	    // create a vertex shader type (GLES20.GL_VERTEX_SHADER)
	    // or a fragment shader type (GLES20.GL_FRAGMENT_SHADER)
	    int shader = GLES20.glCreateShader(type);

	    // add the source code to the shader and compile it
	    GLES20.glShaderSource(shader, shaderCode);
	    GLES20.glCompileShader(shader);

	    return shader;
    }
    
    public void draw( long t ) {
        // Add program to OpenGL ES environment
        GLES20.glUseProgram(program);

        // get handle to vertex shader's vPosition member
        int positionHandle = GLES20.glGetAttribLocation(program, "vPosition");
        // Enable a handle to the triangle vertices
        GLES20.glEnableVertexAttribArray( positionHandle);

        // Prepare the triangle coordinate data
        int vertexStride = 0;
        GLES20.glVertexAttribPointer( positionHandle, COORDS_PER_VERTEX,
                                     GLES20.GL_FLOAT, false,
                                     vertexStride, vertexBuffer);

        // get handle to fragment shader's vColor member
        int colorHandle = GLES20.glGetUniformLocation(program, "vColor");
        // get handle to vertex shader's uniform variable deltaT
        int deltaTHandle = GLES20.glGetUniformLocation ( program, "deltaT");
        // Set color for drawing the triangle
        GLES20.glUniform4fv(colorHandle, 1, color, 0);
        GLES20.glUniform1f( deltaTHandle, (float) t );
        // Draw the triangle
        GLES20.glDrawArrays(GLES20.GL_TRIANGLES, 0, vertexCount);

        // Disable vertex array
        GLES20.glDisableVertexAttribArray(positionHandle);
    }
 }
