package opengl.colortemp;

import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.nio.ShortBuffer;

import android.content.Context;
import java.io.IOException;
import android.util.Log;
import android.opengl.GLES20;

public class Square 
{
  private static String LOG_APP_TAG = "io_tag";	
  private Context context;	  
  private String vsCode = null, fsCode = null;		
  private int program, vertexShader, fragmentShader;
  private FloatBuffer vertexBuffer, vertexTempBuffer;
  private ShortBuffer indexArray;
   
  static float squareCoords[] = { 
    -0.5f, -0.5f, 0.0f,  // v0 - bottom left
     0.5f, -0.5f, 0.0f,  // v1 - bottom right
    -0.5f, 0.5f,  0.0f,  // v2 - top left
     0.5f, 0.5f,  0.0f   // v3 - top right
  };
  
  // Temperature at each vertex
  static float vertexTemp[] = {
	     5.0f,             // v0 cold
	    12.0f,             // v1 cool
	    22.0f,             // v2 warm
	    40.0f              // v3 hot (upper right)
	  };
  
   private short drawOrder[] = { 0, 1, 2, 2, 1, 3 };
  
   // Constructor   
  Square( Context context0){
    context = context0;
	// get shader codes from res/raw/vshader and res/raw/fshader
    vsCode = getShaderCode( GLES20.GL_VERTEX_SHADER );
    fsCode = getShaderCode( GLES20.GL_FRAGMENT_SHADER );
    program = GLES20.glCreateProgram(); // create empty OpenGL ES Program
		  
    vertexShader = loadShader(GLES20.GL_VERTEX_SHADER, vsCode );
    fragmentShader = loadShader(GLES20.GL_FRAGMENT_SHADER, fsCode );
    GLES20.glAttachShader ( program, vertexShader ); // add the vertex shader to program    
    GLES20.glAttachShader(program, fragmentShader); // add the fragment shader to program
    GLES20.glLinkProgram(program);                  // creates OpenGL ES program executables
    GLES20.glUseProgram( program);

    // initialize vertex byte buffer for shape coordinates
    ByteBuffer bb = ByteBuffer.allocateDirect(squareCoords.length * 4);
    bb.order(ByteOrder.nativeOrder());
    // do the for colors
    ByteBuffer bbc = ByteBuffer.allocateDirect( vertexTemp.length * 4 );
    bbc.order(ByteOrder.nativeOrder());
    // create a floating point buffer from the ByteBuffer
    vertexBuffer = bb.asFloatBuffer();
    // add the coordinates to the FloatBuffer
    vertexBuffer.put(squareCoords);
    // set the buffer to read the first coordinate
    vertexBuffer.position(0);
    
    // do the same for colors
    vertexTempBuffer = bbc.asFloatBuffer();
    vertexTempBuffer.put( vertexTemp );
    vertexTempBuffer.position(0);

    // initialize byte buffer for the draw list
    ByteBuffer bbOrder = ByteBuffer.allocateDirect(drawOrder.length * 2);
    bbOrder.order(ByteOrder.nativeOrder());
    indexArray = bbOrder.asShortBuffer();
    indexArray.put(drawOrder);
    indexArray.position(0);
    
  } // Square Constructor
	    
  public void draw() {
    // Add program to OpenGL ES environment
    GLES20.glUseProgram(program);

    // get handle to vertex shader's vPosition and sourceColor member
    int positionHandle = GLES20.glGetAttribLocation(program, "vPosition");
    int vertexTempHandle = GLES20.glGetAttribLocation(program, "vertexTemp");
    int coldColorHandle = GLES20.glGetUniformLocation ( program, "coldColor" );
    int  hotColorHandle = GLES20.glGetUniformLocation ( program, "hotColor" );
    GLES20.glUniform3f ( coldColorHandle, 0.0f, 0.0f, 1.0f);  // blue = cold
    GLES20.glUniform3f ( hotColorHandle, 1.0f, 0.0f, 0.0f);  // red = hot
    
    GLES20.glEnableVertexAttribArray( positionHandle );
    GLES20.glEnableVertexAttribArray( vertexTempHandle );
    GLES20.glVertexAttribPointer( positionHandle, 3,
    		    GLES20.GL_FLOAT, false, 0, vertexBuffer);
    // pass temperature at each vertex to vertex shader
    GLES20.glVertexAttribPointer ( vertexTempHandle, 1, 
    		     GLES20.GL_FLOAT, false, 0, vertexTempBuffer);
     // Draw the square
     GLES20.glDrawElements( GLES20.GL_TRIANGLES, drawOrder.length,
	                GLES20.GL_UNSIGNED_SHORT, indexArray );

     // Disable vertex arrays
     GLES20.glDisableVertexAttribArray( positionHandle );
     GLES20.glDisableVertexAttribArray( vertexTempHandle );
  }   
   
  // Get source code of a shader
  protected String getShaderCode( int type )
  {
    InputStream inputStream = null;
	String str = null;
    try {
      if ( type == GLES20.GL_VERTEX_SHADER )
        inputStream = context.getResources().openRawResource(R.raw.vshader);
      else
        inputStream = context.getResources().openRawResource(R.raw.fshader);
        byte[] reader = new byte[inputStream.available()];;
      while (inputStream.read(reader) != -1) {}
        str = new String ( reader );
     }  catch(IOException e) {
         Log.e(LOG_APP_TAG, e.getMessage());
    }
			 
     return  str;
   }

  public static int loadShader (int type, String shaderCode ) {
    // create a vertex shader type (GLES20.GL_VERTEX_SHADER)
    // or a fragment shader type (GLES20.GL_FRAGMENT_SHADER)
    int shader = GLES20.glCreateShader(type);

    // add the source code to the shader and compile it
    GLES20.glShaderSource(shader, shaderCode);
    GLES20.glCompileShader(shader);

    return shader;
   }    
}
